  ' -----------------------------------------------------------
  ' DFPlayerMini control program.bas
  ' A program for Micromite to communicate with the DFPlayer Mini
  ' module (based on the YX5200-24SS digital audio decoder chip)
  ' to use it as a compact player of MP3 and WAV files.
  ' 
  ' Written by Jim Rowe for Silicon Chip.
  ' Last revision 10/9/2018 at 2:30 pm
  '
  ' Notes:
  ' 1. Communication with the DFPlayer is via the Micromite's
  ' COM2 serial port, which uses pin 9 for TX and pin 10 for RX.
  ' 2. We also use the Micromite's pin 24 for monitoring the DFPlayer's
  ' BUSY pin, so we can tell when the player is busy playing a file
  ' (BUSY = LOW) or stopped (BUSY = HIGH).
  ' 3. The DFPlayer's COM port is pre-programmed to operate at a speed of
  ' 9600 baud (8N1), for sending commands and providing responses. Luckily
  ' the Micromite's COM2 port uses this speed and 8N1 format by default.
  ' 4. The DFPlayer's effective command format is (hex):
  ' 7E FF 06 CM 00 [dh] [dl] EF, where the first 3 bytes (7E FF 06) are
  ' essentially the preamble and the last byte (EF) signifies the end of
  ' the packet. The 'CM' byte is the actual command code, while the following
  ' bytes indicate whether or not any 'feedback' is required (00 = no,
  ' 01 = yes), and then provide any data parameters that may be needed
  ' with that command (dh = high byte, dl = low).

  ' ----------------------------------------------------------
  
  OPTION AUTORUN ON
  OPTION EXPLICIT
  
  DIM AS INTEGER IntFlag = 0     ' flag to show when LCD screen is touched
  DIM AS INTEGER ModOK = 0       ' flag to show if module powered up OK
  DIM AS INTEGER PausFlg = 0     ' flag to show if play deliberately paused
  
  DIM AS STRING Pream$ = CHR$(&H7E) + CHR$(&HFF) + CHR$(&H06) ' preamble
  DIM AS STRING StEnd$ = CHR$(&HEF)  ' end of message
  ' now we define strings for body of commands (CM + 00 + dh + dl)
  ' cmd for Next track
  DIM AS STRING NextCmd$ = CHR$(&H01) + CHR$(&H00) + CHR$(&H00) + CHR$(&H00)
  ' cmd for the Previous track
  DIM AS STRING PrevCmd$ = CHR$(&H02) + CHR$(&H00) + CHR$(&H00) + CHR$(&H00)
  ' cmd specifying the track to be played (0000 - 0BB7)
  DIM AS STRING SpTkCmd$ = CHR$(&H03) + CHR$(&H00) + CHR$(&H00) + CHR$(&H01)
  ' cmd to increase the volume
  DIM AS STRING IncVCmd$ = CHR$(&H04) + CHR$(&H00) + CHR$(&H00) + CHR$(&H00)
  ' cmd to decrease the volume
  DIM AS STRING DecVCmd$ = CHR$(&H05) + CHR$(&H00) + CHR$(&H00) + CHR$(&H00)
  ' cmd to specify the volume (00 - 1E, with 1E as maximum volume)
  DIM AS STRING SpVoCmd$ = CHR$(&H06) + CHR$(&H00) + CHR$(&H00) + CHR$(&H14)
  ' cmd to specify the desired equalisation (00 = Normal, 04 = Classical)
  DIM AS STRING SpEqCmd$ = CHR$(&H07) + CHR$(&H00) + CHR$(&H00) + CHR$(&H00)
  ' cmd to specify the playback mode (00 = normal)
  DIM AS STRING SpPMCmd$ = CHR$(&H08) + CHR$(&H00) + CHR$(&H00) + CHR$(&H00)
  ' cmd to specify the playback source (02 = TF/microSD card)
  DIM AS STRING SpScCmd$ = CHR$(&H09) + CHR$(&H00) + CHR$(&H00) + CHR$(&H02)
  ' cmd to specify normal working
  DIM AS STRING NorWCmd$ = CHR$(&H0B) + CHR$(&H00) + CHR$(&H00) + CHR$(&H00)
  ' cmd to reset the module
  DIM AS STRING RModCmd$ = CHR$(&H0C) + CHR$(&H00) + CHR$(&H00) + CHR$(&H00)
  ' cmd to play or resume play
  DIM AS STRING PBakCmd$ = CHR$(&H0D) + CHR$(&H00) + CHR$(&H00) + CHR$(&H00)
  ' cmd to pause playback
  DIM AS STRING PausCmd$ = CHR$(&H0E) + CHR$(&H00) + CHR$(&H00) + CHR$(&H00)
  ' cmd to play from folder ff (01-63) and track tt (01 - FF)
  DIM AS STRING SpFlCmd$ = CHR$(&H0F) + CHR$(&H00) + CHR$(&H01) + CHR$(&H01)
  ' cmd to set volume adjust (dh = 01 to change, dl = 00 - 1E)
  DIM AS STRING VAdjCmd$ = CHR$(&H10) + CHR$(&H00) + CHR$(&H01) + CHR$(&H14)
  
  CONST TRUE = 1
  CONST FALSE = 0
  Const DBlue = RGB(0,0,128)
  CONST Bone = RGB(255,255,192)
  CONST White = RGB(WHITE)
  CONST Black = RGB(BLACK)
  CONST Red = RGB(RED)
  CONST Green = RGB(GREEN)
  
  SETPIN 15, INTL, TchInt ' call TchInt to set flag when screen touched
  SETPIN 24, DIN          ' declare pin 24 a digital input (for BUSY sensing)
  OPEN "COM2:" AS #2       ' open the Micromite's COM2 port in default mode
  ' ----------------------------------------------------------------------
  ' Pause for 5 seconds to allow module to initialise itself
  Pause 5000
  ' then try reading init confirmation from module
  CheckModule
  IF ModOK = 0 THEN   ' if module not OK, advise and stop program
    PRINT "Module not initialised"
    END
  ELSE
    PRINT "Module present and has initialised"
  END IF
  ' module OK, so now configure the DFPlayer:
  ' set for normal working (whatever that may be)
  PRINT #2, Pream$ + NorWCmd$ + StEnd$
  PRINT "Normal working"
  ' then set playback source to microSD card
  PRINT #2, Pream$ + SpScCmd$ + StEnd$
  PRINT "Playback from microSD card"
  ' then set playback equalisation to 'Normal'
  PRINT #2, Pream$ + SpEqCmd$ + StEnd$
  PRINT "Equalisation set to Normal"
  ' then specify folder 01 & track 01 to be used for playback
  PRINT #2, Pream$ + SpFlCmd$ + StEnd$
  PRINT "Folder 1 & Track 1 to be used for Playback"
  ' then set volume to 20 ( = &H14. Maximum is 30d or &H1E) 
  PRINT #2, Pream$ + VadjCmd$ + StEnd$
  PRINT "Volume set to 20"
  PRINT " "  
  PausFlg = 0   ' clear deliberate pause flag
  ' now show the opening/control screen
  CLS Black
  RBOX 0,0, MM.HRes-2, MM.VRes-2, 5, RGB(Cyan), Black
  TEXT MM.HRes/2, MM.VRes/16, "SILICON CHIP", CM, 1, 2, Red, Black
  TEXT MM.HRes/2, MM.VRes*3/16, "DFPlayer Mini MP3/WAV Audio Player",CM, 1, 1, White, Black
  
  RBOX 4, MM.VRes/4, MM.HRes/2-6, MM.VRes/4-4, 8, Black, Green
  TEXT MM.HRes/4, MM.VRes*3/8, "PLAY", CM, 1, 3, Black, Green
  
  RBOX MM.HRes/2, MM.VRes/4, MM.HRes/2-6, MM.VRes/4-4, 8, Black, Bone
  TEXT MM.HRes*3/4, MM.VRes*3/8, "PAUSE", CM, 1, 3, Red, Bone
   
  RBOX 4, MM.VRes/2, MM.HRes/2-6, MM.VRes/4-4, 8, Black, Red
  TEXT MM.HRes/4, MM.VRes*5/8, "< PREV", CM, 1, 2, White, Red 
   
  RBOX 4, MM.VRes*3/4, MM.HRes/2-6, MM.VRes/4-6, 8, Black, Bone
  TEXT MM.HRes/4, MM.VRes*7/8, "v VOLUME", CM, 1, 2, Black, Bone 
   
  RBOX MM.HRes/2, MM.VRes/2, MM.HRes/2-6, MM.VRes/4-4, 8, Black, Red
  TEXT MM.HRes*3/4, MM.VRes*5/8, "NEXT >", CM, 1,2, White, Red 
   
  RBOX MM.HRes/2, MM.VRes*3/4, MM.HRes/2-6, MM.VRes/4-6, 8, Black, Bone
  TEXT MM.HRes*3/4, MM.VRes*7/8, "VOLUME ^", CM, 1,2, Black, Bone  
  ' -----------------------------------------------------------------
  ' main program loop starts here
  DO
    IF IntFlag = 1 THEN CheckBtn   ' if screen touched, go check & respond
    
    IF PIN(24) = 1 THEN   ' check BUSY-bar line (1 = stopped, 0 = playing)
      IF PausFlg = 0 THEN   'if player has just reached end of track,
        PRINT "Play stopped"  ' print msg
        PRINT #2, Pream$ + NextCmd$ + StEnd$ ' send 'play next' command
        PAUSE 500   ' short pause before continuing
        PRINT "Now playing next file"   ' and confirm now playing again
      ELSE        ' otherwise if deliberately paused (PausFlg = 1)
        PAUSE 100 ' just pause briefly
      END IF
    END IF        ' then continue
    
  LOOP
  
END ' end of main part of program, subroutines follow
  
  ' *****************************************************************
  ' subroutine called when screen is touched, to check whether it has
  ' been touched on either the Play, Pause, Previous, Next, Volume Down
  ' or Volume Up uttons on the LCD (or not, in which case it just returns)
SUB CheckBtn
  IntFlag = 0       ' first clear interrupt flag
  IF TOUCH(Y) < MM.VRes/4 THEN
    EXIT SUB        ' no valid button touch, so just exit
  END IF
  IF TOUCH(X) < MM.HRes/2 THEN    ' if touch was on LH side
    SELECT CASE TOUCH(Y)
      CASE < MM.VRes/2  ' PLAY button touched, so 
        PRINT #2, Pream$ + PBakCmd$ + StEnd$
        PausFlg = 0     ' clear pause flag
        PAUSE 100       ' to allow player to start playing
        IF PIN(24) = 0 THEN ' now check if module is playing or not
        Print "Now playing" ' and if it is, confirm
        ELSE        
          PRINT "NOT playing..."  ' otherwise indicate problem
        END IF
      CASE MM.VRes/2 TO MM.VRes*3/4 ' <PREV button touched, so send 'previous' cmd
        PRINT #2, Pream$ + PrevCmd$ + StEnd$
        PRINT "Now playing previous file" ' and confirm
      CASE > MM.VRes*3/4  ' vVOLUME button touched, so send 'decrease volume' cmd
        PRINT #2, Pream$ + DecVCmd$ + StEnd$
        PRINT "Volume decreased"  ' and confirm
    end select
  ELSE    ' touch must have been on lower RH side
    SELECT CASE TOUCH(Y)
      CASE < MM.VRes/2  ' PAUSE button touched, so send 'pause' cmd
        PRINT #2, Pream$ + PausCmd$ + StEnd$
        PausFlg = 1     ' set the pause flag to show deliberate pause
        Print "Playing paused" ' and confirm
      CASE MM.VRes/2 TO MM.VRes*3/4 ' NEXT> button touched, so send 'next' cmd
        PRINT #2, Pream$ + NextCmd$ + StEnd$
        PRINT "Now playing next file"   ' and confirm
      CASE > MM.VRes*3/4  ' VOLUME^ button touched, so send increase volume cmd
        PRINT #2, Pream$ + IncVCmd$ + StEnd$
        PRINT "Volume increased"  ' and confirm
    end select
  END IF
END SUB             ' before returning  
  ' -----------------------------------------------------------------
  ' subroutine to set IntFlg when screen is touched
SUB TchInt
  IntFlag = 1
END SUB
  ' -------------------------------------------------------------------
  ' subroutine to check for a response from the DLPlayer module
SUB CheckModule
  IF EOF(#2) = 0 THEN  ' if no chars from module as yet,
    ModOK = 0          ' clear ModOK and exit
    EXIT SUB
  ELSE
    ModOK = 1               ' but if chars have arrived, set ModOK = 1
  END IF
END SUB 
  ' *******************************************************************
